/*
    Airodump CSV Tools v0.4
    Merges and parses Airodump CSV files and outputs them to CSV, HTML, or text.
    Copyright (C) 2013-2015 Christopher Bolduc

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#define CRLF "\r\n"

/* I decided it was easier to write my own than use the library.
 * Used by the program to compare dates, not for output.
 */
typedef struct datetime {
  int year;
  int month;
  int day;
  int hour;
  int minute;
  int second;
} datetime;

typedef struct gps {
  datetime dt;
  double lat;
  double lon;

  struct gps *next;
} gps;

// Linked list of Access Points
typedef struct ap {
  char bssid[80];
  char vendor[80];
  char first_time_seen[80];
  char last_time_seen[80];
  char channel[80];
  char speed[80];
  char privacy[80];
  char cipher[80];
  char authentication[80];
  int power;
  char beacons[80];
  char ivs[80];
  char lan_ip[80];
  char id_length[80];
  char essid[80];
  char key[80];
  // Filename is still left in here from older versions of this program.
  char fileName[80];
  char desc[80];
  int oldPower;

  datetime time1;
  datetime time2;
  int maxPwrLevel;
  char maxPwrTime[80];
  datetime mpTime;
  double lat;
  double lon;
  int new;
  int old;
  struct ap *next;
} ap;

// Linked list of End Devices
typedef struct enddev {
  char station_mac[80];
  char vendor[80];
  char first_time_seen[80];
  char last_time_seen[80];
  int power;
  char packets[80];
  char bssid[80];
  char essid[80];
  char probed_essids[255];
  char fileName[80];
  char desc[80];
  int oldPower;

  datetime time1;
  datetime time2;
  int maxPwrLevel;
  char maxPwrTime[80];
  datetime mpTime;
  double lat;
  double lon;
  int new;
  int old;
  struct enddev *next;
} enddev;

typedef struct macdb {
  char mac[18];
  char vendor[80];

  struct macdb *next;
} macdb;

typedef struct devset {
  ap *s;
  enddev *e;
} devset;

// Prototypes
char *str_replace(char *s, char old, char new);
int strToTime (datetime *dest, char *str);
int compareDates (datetime *d1, datetime *d2);
void free_ap (ap *s);
void free_enddev (enddev *e);
void free_gps (gps *g);
long getEssid(char *currWord, char *buffer, long i, long lSize);
long getWord(char *currWord, char *buffer, long i, long lSize);
ap *findApByBSSID (ap *s, char *key);
char *findVendorByMAC (macdb * m, char * key);
devset readCSVFile (char * fileName, ap *firstAp, enddev *firstEnddev, const int lastFile);
void readMacDB (char * fileName);
void readKnownMacs (char * fileName);
void addGPSInfo (ap *firstap, enddev *firsted, gps *firstg);
gps *readGPSFile (ap *firstap, enddev *firsted, FILE *f);
void readAPPowerFromFile (ap *first, FILE *f);
void readEnddevPowerFromFile (enddev *first, FILE *f);
void printAPPowerToFile (ap *a, FILE *f);
void printAPPowerToFileRec (ap *a, FILE *f);
void printEndDevicesPowerToFile (enddev *e, FILE *f);
void printEndDevicesPowerToFileRec (enddev *e, FILE *f);
void printAPToFileText (ap *a, FILE *f);
//void printAPsToFileText (ap *a, FILE *f);
void printEndDeviceToFileText (enddev *e, FILE *f);
//void printEndDevicesToFileText (enddev *e, FILE *f);
void printAPToFileCSV (ap *a, FILE *f);
void printAPsToFileRec (ap *a);
void printEndDeviceToFileCSV (enddev *e, FILE *f);
void printEndDevicesToFileRec (enddev *e);
void printAPToFileHTML (ap *a, FILE *f);
//void printAPsToFileHTML (ap *a, FILE *f);
void printEndDeviceToFileHTML (enddev *e, FILE *f);
//void printEndDevicesToFileHTML (enddev *e, FILE *f);

// Boolean Globals
int onlyAddCommon;
int onlyAddNew;
int onlyAddOld;
int onlyShowKnown;
int deltaSpecified;
int text_brief;

// Other Globals
int verbosity;
int numInputFiles;
int minPower;
int maxPower;
int minPowerDelta;
FILE *kmlFile, *textFile, *htmlFile, *csvFile;
macdb *mac_database, *known_macs;

// Replace a character in a string
char *str_replace(char *s, char old, char new) {
  char *p = s;

  while(*p) {
    if(*p == old)
    *p = new;
    ++p;
  }

  return s;
}

int strToTime (datetime *dest, char *str) {
  return sscanf (str, "%d-%d-%d %d:%d:%d",
      &(dest->year),
      &(dest->month),
      &(dest->day),
      &(dest->hour),
      &(dest->minute),
      &(dest->second));
}

// Returns 1 if d1 > d2, -1 if d2 > d1, 0 if equal
int compareDates (datetime *d1, datetime *d2) {
  if (d1->year > d2->year) return 1;
  if (d1->year < d2->year) return -1;
  if (d1->month > d2->month) return 1;
  if (d1->month < d2->month) return -1;
  if (d1->day > d2->day) return 1;
  if (d1->day < d2->day) return -1;
  if (d1->hour > d2->hour) return 1;
  if (d1->hour < d2->hour) return -1;
  if (d1->minute > d2->minute) return 1;
  if (d1->minute < d2->minute) return -1;
  if (d1->second > d2->second) return 1;
  if (d1->second < d2->second) return -1;
  return 0;
}

// Free the linked list of APs
void free_ap (ap *s) {
  if (s == NULL) return;
  free_ap(s->next);
  free(s);
}

// Free the linked list of Enddevs
void free_enddev (enddev *e) {
  if (e == NULL) return;
  free_enddev(e->next);
  free(e);
}

// Free the linked list of GPS times and coordinates
void free_gps (gps *g) {
  if (g == NULL) return;
  free_gps(g->next);
  free(g);
}

// This is a separate function from getWord in case the ESSID has commas in it
long getEssid(char *currWord, char *buffer, long i, long lSize) {
  long lastComma = 0;
  int j = i;
  // Skip leading whitespace
  while (i < lSize && (buffer[i] == ' ' || buffer[i] == '\t')) i++;

  while (j < lSize-1) {
    if (buffer[j] == '\r' && buffer[j+1] == '\n') {
      break;
    }
    if (buffer[j] == ',') {
      lastComma = j;
    }
    j++;
  }

  j=0;
  while (i < lastComma && i < lSize-1) {
    currWord[j] = buffer[i];
    i++;
    j++;
  }
  i++;
  j++;
  currWord[j] = '\0';
  return i;
}

// Reads one word from the input file (buffer) and places it into currWord
// Returns the position in buffer
long getWord(char *currWord, char *buffer, long i, long lSize) {
  long j=0;

  // Skip leading whitespace
  while (i < lSize && (buffer[i] == ' ' || buffer[i] == '\t')) i++;

  while (i < lSize-1) {
    if (buffer[i] == '\r' && buffer[i+1] == '\n') {
      // Why, Microsoft?
      i += 2;
      currWord[j] = '\0';
//      printf ("getWord: %s\n", currWord);
      return i;
    }
    if (buffer[i] == ',' || buffer[i] == '\n') {
      currWord[j] = '\0';
//      printf ("getWord: %s\n", currWord);
      i++;
      return i;
    }
    currWord[j] = buffer[i];
    i++;
    j++;
  }
  return i;
}

// Finds an AP in the linked list given the BSSID (key)
ap *findApByBSSID (ap *s, char *key) {
//  printf ("Got next: %d\n", temp);
//  printf ("Next BSSID: %s\n", s->next->bssid);
  if (!s) return NULL;
  if (strcmp (s->bssid, key) == 0) return s;
//  if (s->next == NULL) return NULL;
  return findApByBSSID(s->next, key);
}

char *findVendorByMAC (macdb * m, char * key) {
//  printf ("Searching for: %s\n", key);
  if (!m) return "";
  if (strcmp(m->mac, key) == 0) return m->vendor;
//  if (m->next == NULL) return "";
  return findVendorByMAC(m->next, key);
}

// Finds an Enddev in the linked list given the Station MAC (key)
enddev *findEnddevByMAC (enddev *e, char *key) {
  if (strcmp (e->station_mac, key) == 0) return e;
  if (e->next == NULL) return NULL;
  return findEnddevByMAC(e->next, key);
}

void printAPAlert (ap *a) {
  printf ("BSSID %s seen at %s", a->bssid, a->last_time_seen);
}

// Prints a single AP to a file with its max power level 
void printAPPowerToFile (ap *a, FILE *f) {
//  printf ("%s, %d, %s%s", a->bssid, a->maxPwrLevel, a->maxPwrTime, CRLF);
  fprintf (f, "%s, %d, %s%s", a->bssid, a->maxPwrLevel, a->maxPwrTime, CRLF);
}

void printAPPowerToFileRec (ap *a, FILE *f) {
  int result;
  int i;
  ap *currAp = a;

  if (a == NULL) {
    fprintf(f, "%s", CRLF);
    return;
  }
  printAPPowerToFile (a, f);

  result = ferror (f);
  if (result) {
    printf ("printAPsPowersToFile fprintf returned error: %d\n", result);
    return;
  }
  printAPPowerToFileRec (a->next, f);
}

void printEndDevicesPowerToFile (enddev *e, FILE *f) {
  fprintf (f, "%s, %d, %s%s", e->station_mac, e->maxPwrLevel, e->maxPwrTime, CRLF);
//  printf ("%s, %d, %s%s", e->station_mac, e->maxPwrLevel, e->maxPwrTime, CRLF);
}

// Prints power levels from a linked list of Enddevs (e) to a file (f)
void printEndDevicesPowerToFileRec (enddev *e, FILE *f) {
  int result;
  int i;
  enddev *curr = e;

  if (e == NULL) {
    fprintf(f, "%s", CRLF);
    return;
  }
  printEndDevicesPowerToFile (e, f);

  result = ferror (f);
  if (result) {
    printf ("printEndDevicesPowersToFile fprintf returned error: %d\n", result);
    return;
  }
  printEndDevicesPowerToFileRec (e->next, f);
}

// sz is populated by the function
char *readFileToString(FILE *pFile, long *sz) {
  long lSize;
  char *buffer;
  size_t result;

  // obtain file size
  fseek (pFile, 0, SEEK_END);
  lSize = ftell (pFile);
  rewind (pFile);

  // allocate memory to contain the whole file
  buffer = (char*) malloc(sizeof(char) * lSize);
  if (buffer == NULL) {
    fputs ("Memory error\n", stderr);
    exit(2);
  }

  // copy the file into the buffer
  result = fread (buffer, 1, lSize, pFile);
  if (result != lSize) {
    fputs ("Reading error\n", stderr);
    exit(3);
  }
//  fclose(pFile);
  *sz = lSize;
  return buffer;
}

void readAPPowerFromFile (ap *first, FILE *f) {
  ap *curr;
  char bssid[80];
  char pwrbuf[80];
  char time[80];
  char *buffer;
  int power;
  int result;
  long lSize;

  if (first == NULL) return;
  buffer = readFileToString(f, &lSize);
  long i=0;
  int j=0;
  while (i<lSize) {
    j=0;
    while (i<lSize) {
      if (buffer[i] == ',') {
        bssid[j] = '\0';
        i++;
        break;
      }
      else if (buffer[i] == '\r' || buffer[i] == '\n') {
        printf ("Error: unexpected EOL for bssid %s\n", bssid);
        return;
      }
      else
        bssid[j] = buffer[i];
      i++;
      j++;
    }
//    printf ("bssid: %s ", bssid);
    while (buffer[i] == ' ') i++;
    j=0;
    while (i<lSize) {
      if (buffer[i] == ',')  {
        pwrbuf[j] = '\0';
        i++;
        break;
      }
      else if (buffer[i] == '\r' || buffer[i] == '\n') {
        printf ("Error: unexpected EOL for bssid %s\n", bssid);
        return;
      }
      else
        pwrbuf[j] = buffer[i];
      i++;
      j++;
    }
    power = atoi(pwrbuf);
//    printf ("pwr: %d ", power);
    j=0;
    while (buffer[i] == ' ') i++;
    while (i<lSize) {
      if (buffer[i] == ',' || buffer[i] == '\r' || buffer[i] == '\n') {
        time[j] = '\0';
        i++;
        break;
      }
      else
        time[j] = buffer[i];
      i++;
      j++;
    }
    while (buffer[i] == ' ' || buffer[i] == '\r' || buffer[i] == '\n') i++;
//    printf ("time: %s\n", time);
    curr = findApByBSSID (first, bssid);
    if (curr != NULL) {
      curr->maxPwrLevel = power;
      strcpy(curr->maxPwrTime, time);
    }
  }

  free (buffer);
}

void readEnddevPowerFromFile (enddev *first, FILE *f) {
  enddev *curr;
  char mac[80];
  char pwrbuf[80];
  char time[80];
  char *buffer;
  int power;
  int result;
  long lSize;

  if (first == NULL) return;
  buffer = readFileToString(f, &lSize);
  long i=0;
  int j=0;
  while (i<lSize) {
    j=0;
    while (i<lSize) {
      if (buffer[i] == ',') {
        mac[j] = '\0';
        i++;
        break;
      }
      else if (buffer[i] == '\r' || buffer[i] == '\n') {
        printf ("Error: unexpected EOL for mac %s\n", mac);
        return;
      }
      else
        mac[j] = buffer[i];
      i++;
      j++;
    }
//    printf ("mac: %s ", mac);
    while (buffer[i] == ' ') i++;
    j=0;
    while (i<lSize) {
      if (buffer[i] == ',')  {
        pwrbuf[j] = '\0';
        i++;
        break;
      }
      else if (buffer[i] == '\r' || buffer[i] == '\n') {
        printf ("Error: unexpected EOL for mac %s\n", mac);
        return;
      }
      else
        pwrbuf[j] = buffer[i];
      i++;
      j++;
    }
    power = atoi(pwrbuf);
//    printf ("pwr: %d ", power);
    j=0;
    while (buffer[i] == ' ') i++;
    while (i<lSize) {
      if (buffer[i] == ',' || buffer[i] == '\r' || buffer[i] == '\n') {
        time[j] = '\0';
        i++;
        break;
      }
      else
        time[j] = buffer[i];
      i++;
      j++;
    }
    while (buffer[i] == ' ' || buffer[i] == '\r' || buffer[i] == '\n') i++;
//    printf ("time: %s\n", time);
    curr = findEnddevByMAC (first, mac);
    if (curr != NULL) {
      curr->maxPwrLevel = power;
      strcpy(curr->maxPwrTime, time);
    }
  }

  free (buffer);
}

gps *readGPSFile (ap *firstap, enddev *firsted, FILE *f) {
  gps *g, *g1, *gprev;
  int result;

  gprev = g = g1 = (gps *) malloc (sizeof(gps));

  while (1) {
    result = fscanf (f, "%d-%d-%d %d:%d:%d, %lf, %lf\r\n", &(g->dt.year), &(g->dt.month), &(g->dt.day), &(g->dt.hour), &(g->dt.minute), &(g->dt.second), &(g->lat), &(g->lon));
    if (result == EOF) {
      gprev->next = NULL;
      free(g);
      return g1;
    }
    g->next = (gps *) malloc (sizeof(gps));
    gprev = g;
    g = g->next;
  }
  return g1;
}

void addGPSInfo (ap *firstap, enddev *firsted, gps *firstg) {
  ap *currap = firstap;
  enddev *curred = firsted;
  gps *g = firstg;
  int result;
  int badtime = 0;

  while (currap != NULL) {
//    printf ("Read AP GPS info for: %s\n", currap->bssid);
    result = strToTime (&(currap->mpTime), currap->maxPwrTime);
    if (result == EOF) {
      printf ("Error reading GPS file\n");
      printf ("Max power time: %s\n", currap->maxPwrTime);
      return;
    }
    if (compareDates (&(currap->mpTime), &(g->dt)) == -1) {
      // Skip this AP, too old
      currap = currap->next;
      continue;
    }

    badtime = 0;
    while (compareDates (&(currap->mpTime), &(g->dt)) == 1) {
      // If compareDates returns 0 or -1, the current AP's date is less than or equal to
      // the date in the GPS file we are checking against.
//      printf ("Is %d:%d:%d > %d:%d:%d?\n", currap->mpTime.hour, currap->mpTime.minute, currap->mpTime.second, g->dt.hour, g->dt.minute, g->dt.second);
      g = g->next;
      if (g == NULL) {
        // Date not in range
        badtime = 1;
        break;
      }
    }
    currap->lat = badtime ? 0 : g->lat;
    currap->lon = badtime ? 0 : g->lon;
    g = firstg;
    currap = currap->next;
  }
  while (curred != NULL) {
//    printf ("Reading Station info for: %s\n", curred->station_mac);
    result = strToTime (&(curred->mpTime), curred->maxPwrTime);
    if (result == EOF) {
      printf ("Error reading station info for %s\n", curred->station_mac);
      printf ("Max power time: %s\n", curred->maxPwrTime);
      return;
    }
    if (compareDates (&(curred->mpTime), &(g->dt)) == -1) {
      // Skip this end device, too old
      curred = curred->next;
      continue;
    }

    badtime = 0;
    while (compareDates (&(curred->mpTime), &(g->dt)) == 1) {
      g = g->next;
      if (g == NULL) {
        // Date not in range
        badtime = 1;
        break;
      }
    }
    curred->lat = badtime ? 0 : g->lat;
    curred->lon = badtime ? 0 : g->lon;
    g = firstg;
    curred = curred->next;
  }
}

// Prints a single AP (a) to a file (f)
void printAPToFileKML (ap *a, FILE *f) {
  if (a->power < minPower) return;
  if (a->power > maxPower) return;
  if (a->lat == 0.0) return; // no GPS data
  if (onlyShowKnown && strcmp(a->desc, "") == 0) return;
  fprintf(f, "<Placemark>%s"
    "<name>%s (%s)</name>%s"
    "<description>%s", CRLF, a->essid, str_replace(a->vendor, '&', ' '), CRLF, CRLF);
  fprintf (f, "Description: %s%s", a->desc, CRLF);
  fprintf (f, "BSSID: %s%s", a->bssid, CRLF);
  fprintf (f, "Vendor: %s%s", str_replace(a->vendor, '&', ' '), CRLF);
  fprintf (f, "First time seen: %s%s", a->first_time_seen, CRLF);
  fprintf (f, "Last time seen: %s%s", a->last_time_seen, CRLF);
  fprintf (f, "Channel: %s%s", a->channel, CRLF);
  fprintf (f, "Speed: %s%s", a->speed, CRLF);
  fprintf (f, "Privacy: %s%s", a->privacy, CRLF);
  fprintf (f, "Cipher: %s%s", a->cipher, CRLF);
  fprintf (f, "Authentication: %s%s", a->authentication, CRLF);
  fprintf (f, "Power: %d%s", a->power, CRLF);
  fprintf (f, "Previous Power: %d%s", a->oldPower, CRLF);
  fprintf (f, "Beacons: %s%s", a->beacons, CRLF);
  fprintf (f, "IVs: %s%s", a->ivs, CRLF);
  fprintf (f, "LAN IP: %s%s", a->lan_ip, CRLF);
  fprintf (f, "ID-length: %s%s", a->id_length, CRLF);
  fprintf (f, "ESSID: %s%s", a->essid, CRLF);
  fprintf (f, "Key: %s%s", a->key, CRLF);
  fprintf (f, "Max Power: %d%s", a->maxPwrLevel, CRLF);
  fprintf (f, "Max Power Time: %s%s", a->maxPwrTime, CRLF);
  fprintf (f, "File: %s%s%s", a->fileName, CRLF, CRLF);

  fprintf (f, "</description>%s"
    "<Point>%s"
      "<coordinates>%lf,%lf</coordinates>%s"
    "</Point>%s"
  "</Placemark>%s", CRLF, CRLF, a->lon, a->lat, CRLF, CRLF, CRLF);

}

// Prints a single Enddev (e) to a file (f)
void printEndDeviceToFileKML (enddev *e, FILE *f) {
  int powerDelta = e->power - e->oldPower;
//  if (powerDelta < 0) powerDelta = -powerDelta;
  if (e->power < minPower) return;
  if (e->power > maxPower) return;
  if (e->lat == 0.0) return; // no GPS data
  if (onlyShowKnown && strcmp(e->desc, "") == 0) return;
  if (deltaSpecified) {
    if (e->oldPower >= -1 || powerDelta <= minPowerDelta) return;
  }
  fprintf(f, "<Placemark>%s"
    "<name>%s (%s)</name>%s"
    "<description>%s", CRLF, e->station_mac, str_replace(e->vendor, '&', ' '), CRLF, CRLF);
  fprintf (f, "Description: %s%s", e->desc, CRLF);
  fprintf (f, "Station MAC: %s%s", e->station_mac, CRLF);
  fprintf (f, "Vendor: %s%s", str_replace(e->vendor, '&', ' '), CRLF);
  fprintf (f, "First time seen: %s%s", e->first_time_seen, CRLF);
  fprintf (f, "Last time seen: %s%s", e->last_time_seen, CRLF);
  fprintf (f, "Power: %d%s", e->power, CRLF);
  fprintf (f, "Previous Power: %d%s", e->oldPower, CRLF);
  fprintf (f, "Packet count: %s%s", e->packets, CRLF);
  fprintf (f, "BSSID: %s%s", e->bssid, CRLF);
  fprintf (f, "ESSID: %s%s", e->essid, CRLF);
  fprintf (f, "Probed ESSIDs: %s%s", e->probed_essids, CRLF);
  fprintf (f, "Max Power: %d%s", e->maxPwrLevel, CRLF);
  fprintf (f, "Max Power Time: %s%s", e->maxPwrTime, CRLF);
  fprintf (f, "File %s%s%s", e->fileName, CRLF, CRLF);
  fprintf (f, "</description>%s"
    "<Point>%s"
      "<coordinates>%lf,%lf</coordinates>%s"
    "</Point>%s"
  "</Placemark>%s", CRLF, CRLF, e->lon, e->lat, CRLF, CRLF, CRLF);
}

// Prints a single AP (a) to a file (f)
void printAPToFileHTML (ap *a, FILE *f) {
  if (a->power < minPower) return;
  if (a->power > maxPower) return;
  if (onlyShowKnown && strcmp(a->desc, "") == 0) return;
  fprintf (f, "<tr><td>%s</td><td>%s</td><td>%s</td><td>%s</td><td>%s</td><td>%s</td><td>%s</td><td>%s</td>"
    "<td>%s</td><td>%d</td><td>%s</td><td>%s</td><td>%s</td><td>%s</td><td>%s</td><td>%s</td><td>%s</td></tr>%s",
    a->bssid, a->vendor, a->first_time_seen, a->last_time_seen, a->channel, a->speed, a->privacy, a->cipher,
    a->authentication, a->power, a->beacons, a->ivs, a->lan_ip, a->id_length, a->essid, a->key, a->desc,
    CRLF);
}
/*
// Prints all of the APs in the linked list (a) to a file (f)
void printAPsToFileHTML (ap *a, FILE *f) {
  int result;
  int i;
  ap *currAp = a;

  if (a == NULL) {
    fprintf(f, "%s", CRLF);
    return;
  }
  if (onlyAddNew) {
    if (a->new) printAPToFileHTML (a, f);
  } else if (onlyAddOld) {
    if (a->old) printAPToFileHTML(a, f);
  } else {
    printAPToFileHTML (a, f);
  }

  result = ferror (f);
  if (result) {
    printf ("printAPsToFileHTML fprintf returned error: %d\n", result);
    return;
  }
  printAPsToFileHTML (a->next, f);
}
*/
// Prints a single Enddev (e) to a file (f)
void printEndDeviceToFileHTML (enddev *e, FILE *f) {
  if (e->power < minPower) return;
  if (e->power > maxPower) return;
  if (onlyShowKnown && strcmp(e->desc, "") == 0) return;
  fprintf (f, "<tr><td>%s</td><td>%s</td><td>%s</td><td>%s</td><td>%d</td><td>%s</td><td>%s</td><td>%s</td><td>%s</td><td>%s</td></tr>%s", e->station_mac, e->vendor, e->first_time_seen,
    e->last_time_seen, e->power, e->packets, e->bssid, e->essid, e->probed_essids, e->desc, CRLF);
}
/*
// Prints a linked list of Enddevs (e) to a file (f)
void printEndDevicesToFileHTML (enddev *e, FILE *f) {
  int result;
  int i;
  enddev *curr = e;

  if (e == NULL) {
    fprintf(f, "%s", CRLF);
    return;
  }
  if (onlyAddNew) {
    if (e->new) printEndDeviceToFileHTML (e, f);
  } else if (onlyAddOld) {
    if (e->old) printEndDeviceToFileHTML (e, f);
  } else {
    printEndDeviceToFileHTML (e, f);
  }

  result = ferror (f);
  if (result) {
    printf ("printEndDevicesToFileHTML fprintf returned error: %d\n", result);
    return;
  }
  printEndDevicesToFileHTML (e->next, f);
}
*/

// Prints a single AP (a) to a file (f)
void printAPToFileCSV (ap *a, FILE *f) {
  if (a->power < minPower) return;
  if (a->power > maxPower) return;
  fprintf (f, "%s, %s, %s, %s, %s, %s, %s, %s, %d, %s, %s, %s, %s, %s, %s%s",
    a->bssid, a->first_time_seen, a->last_time_seen, a->channel, a->speed, a->privacy, a->cipher,
    a->authentication, a->power, a->beacons, a->ivs, a->lan_ip, a->id_length, a->essid, a->key, CRLF);
}

// Prints all of the APs in the linked list (a) to a file (f)
void printAPsToFileRec (ap *a) {
  int result;
  int i;
  int skip = 1;
  ap *currAp = a;

  if (a == NULL) {
    fprintf(csvFile, "%s", CRLF);
    return;
  }
  if (onlyAddNew) {
    if (a->new) skip = 0;
  } else if (onlyAddOld) {
    if (a->old) skip = 0;
  } else {
    skip = 0;
  }

  if (!skip) {
     printAPToFileCSV (a, csvFile);
     printAPToFileText (a, textFile);
     printAPToFileHTML (a, htmlFile);
     if (kmlFile) printAPToFileKML (a, kmlFile);
  }

/*
  result = ferror (f);
  if (result) {
    printf ("printAPsToFileCSV fprintf returned error: %d\n", result);
    return;
  }
 */
  printAPsToFileRec (a->next);
}

// Prints a single Enddev (e) to a file (f)
void printEndDeviceToFileCSV (enddev *e, FILE *f) {
  if (e->power < minPower) return;
  if (e->power > maxPower) return;
  fprintf (f, "%s, %s, %s, %d, %s, %s, %s%s", e->station_mac, e->first_time_seen,
    e->last_time_seen, e->power, e->packets, e->bssid, e->probed_essids, CRLF);
}

// Prints a linked list of Enddevs (e) to files (text, html, csv)
void printEndDevicesToFileRec (enddev *e) {
  int result;
  int i;
  int skip = 1;
  enddev *curr = e;

  if (e == NULL) {
    fprintf(csvFile, "%s", CRLF);
    return;
  }
  if (onlyAddNew) {
    if (e->new) skip = 0;
  } else if (onlyAddOld) {
    if (e->old) skip = 0;
  } else {
    skip = 0;
  }

  if (!skip) {
     printEndDeviceToFileCSV (e, csvFile);
     printEndDeviceToFileText (e, textFile);
     printEndDeviceToFileHTML (e, htmlFile);
     if (kmlFile) printEndDeviceToFileKML (e, kmlFile);
  }

/*
  result = ferror (f);
  if (result) {
    printf ("printEndDevicesToFileCSV fprintf returned error: %d\n", result);
    return;
  }
*/
  printEndDevicesToFileRec (e->next);
}

// Prints a single ap (a) to a file (f)
void printAPToFileText (ap *a, FILE *f) {
  int powerDelta = a->power - a->oldPower;
  if (a->power < minPower) return;
  if (a->power > maxPower) return;
  if (deltaSpecified) {
    if (a->oldPower >= -1 || powerDelta <= minPowerDelta) return;
  }
  if (onlyShowKnown && strcmp(a->desc, "") == 0) return;
  if (text_brief) {
    fprintf (f, "AP:  %s (%s) ESSID: %s PWR: %d DESC: %s\n", a->bssid, a->vendor, a->essid, a->power, a->desc);
    return;
  }

  fprintf (f, "BSSID: %s%s", a->bssid, CRLF);
  fprintf (f, "Vendor: %s%s", a->vendor, CRLF);
  fprintf (f, "First time seen: %s%s", a->first_time_seen, CRLF);
  fprintf (f, "Last time seen: %s%s", a->last_time_seen, CRLF);
  fprintf (f, "Channel: %s%s", a->channel, CRLF);
  fprintf (f, "Speed: %s%s", a->speed, CRLF);
  fprintf (f, "Privacy: %s%s", a->privacy, CRLF);
  fprintf (f, "Cipher: %s%s", a->cipher, CRLF);
  fprintf (f, "Authentication: %s%s", a->authentication, CRLF);
  fprintf (f, "Power: %d%s", a->power, CRLF);
  fprintf (f, "Previous Power: %d%s", a->oldPower, CRLF);
  fprintf (f, "Beacons: %s%s", a->beacons, CRLF);
  fprintf (f, "IVs: %s%s", a->ivs, CRLF);
  fprintf (f, "LAN IP: %s%s", a->lan_ip, CRLF);
  fprintf (f, "ID-length: %s%s", a->id_length, CRLF);
  fprintf (f, "ESSID: %s%s", a->essid, CRLF);
  fprintf (f, "Key: %s%s", a->key, CRLF);
  fprintf (f, "Max Power: %d%s", a->maxPwrLevel, CRLF);
  fprintf (f, "Max Power Time: %s%s", a->maxPwrTime, CRLF);
  fprintf (f, "Latitude: %lf%s", a->lat, CRLF);
  fprintf (f, "Longitude: %lf%s", a->lon, CRLF);
  fprintf (f, "File: %s%s", a->fileName, CRLF);
  fprintf (f, "Description: %s%s%s", a->desc, CRLF, CRLF);
}
/*
// Prints a linked list of APs (a) to a file (f)
void printAPsToFileText (ap *a, FILE *f) {
  int i, result;
  ap *currAp = a;

  if (a == NULL) return;
  if (onlyAddNew) {
    if (a->new) printAPToFileText (a, f);
  } else if (onlyAddOld) {
    if (a->old) printAPToFileText (a, f);
  } else {
    printAPToFileText (a, f);
  }

  result = ferror (f);
  if (result) {
    printf ("printAPsToFileText fprintf returned error: %d\n", result);
    return;
  }
  printAPsToFileText (a->next, f);
}
*/
// Prints a single Enddev (e) to a file (f)
void printEndDeviceToFileText (enddev *e, FILE *f) {
  int powerDelta = e->power - e->oldPower;
  if (e->power < minPower) return;
  if (e->power > maxPower) return;
  if (onlyShowKnown && strcmp(e->desc, "") == 0) return;
  if (deltaSpecified) {
    if (e->oldPower >= -1 || powerDelta <= minPowerDelta) return;
  }

  if (text_brief) {
    fprintf (f, "STA: %s (%s) ESSID: %s PWR: %d DESC: %s\n", e->station_mac, e->vendor, e->essid, e->power, e->desc);
    return;
  }
  fprintf (f, "Station MAC: %s%s", e->station_mac, CRLF);
  fprintf (f, "Vendor: %s%s", e->vendor, CRLF);
  fprintf (f, "First time seen: %s%s", e->first_time_seen, CRLF);
  fprintf (f, "Last time seen: %s%s", e->last_time_seen, CRLF);
  fprintf (f, "Power: %d%s", e->power, CRLF);
  fprintf (f, "Previous Power: %d%s", e->oldPower, CRLF);
  fprintf (f, "Packet count: %s%s", e->packets, CRLF);
  fprintf (f, "BSSID: %s%s", e->bssid, CRLF);
  fprintf (f, "ESSID: %s%s", e->essid, CRLF);
  fprintf (f, "Probed ESSIDs: %s%s", e->probed_essids, CRLF);
  fprintf (f, "Max Power: %d%s", e->maxPwrLevel, CRLF);
  fprintf (f, "Max Power Time: %s%s", e->maxPwrTime, CRLF);
  fprintf (f, "Latitude: %lf%s", e->lat, CRLF);
  fprintf (f, "Longitude: %lf%s", e->lon, CRLF);
  fprintf (f, "File: %s%s", e->fileName, CRLF);
  fprintf (f, "Description: %s%s%s", e->desc, CRLF, CRLF);
}

/*
// Prints a linked list of Enddevs (e) to a file (f)
void printEndDevicesToFileText (enddev *e, FILE *f) {
  int i, result;
  enddev *curr = e;

  if (e == NULL) return;
  if (onlyAddNew) {
    if (e->new) printEndDeviceToFileText (e, f);
  } else if (onlyAddOld) {
    if (e->old) printEndDeviceToFileText (e, f);
  } else {
    printEndDeviceToFileText (e, f);
  }

  result = ferror (f);
  if (result) {
    printf ("printEndDevicesToFileText fprintf returned error: %d\n", result);
    return;
  }
  printEndDevicesToFileText (e->next, f);
}
*/

// Read the vendor MAC address database into a linked list
void readMacDB (char * fileName) {
  FILE *pFile;
  long lSize;
  char buffer[120];
  char *mac, *vendor;
  macdb *curr_node;
  int i;

  pFile = fopen (fileName, "r");

  if (pFile == NULL) {
   printf ("readMacDB - Error opening file: %s\n", fileName);
   exit(1);
  }

  while (fgets (buffer, 120, pFile) != NULL) {
    if (mac_database == NULL) {
      mac_database = (macdb *) malloc (sizeof(macdb));
      mac_database->next = NULL;
      curr_node = mac_database;
    } else {
      curr_node->next = (macdb *) malloc (sizeof(macdb));
      curr_node = curr_node->next;
      curr_node->next = NULL;
    }
    mac = curr_node->mac;
    vendor = curr_node->vendor;
    memcpy (mac, buffer, 8);
    mac[8] = '\0';
    for (i=0; i < 8; i++) {
      if (mac[i] == '-') mac[i] = ':';
    }
    memcpy (vendor, buffer + 18, 80);
    for (i=0; i < 80; i++) {
      if (vendor[i] == '\n') vendor[i] = '\0';
    }
  }
}

// Reads a CSV list of known MAC addresses (user-generated)
// So they can be placed next to the AP/station
// "Vendor" in this case is the user-generated comment
void readKnownMacs (char * fileName) {
  FILE *pFile;
  long lSize;
  char buffer[120];
  char *mac, *vendor;
  macdb *curr_node;
  int i;

  pFile = fopen (fileName, "r");

  if (pFile == NULL) {
   printf ("readKnownMacs - Error opening file: %s\n", fileName);
   exit(1);
  }

  while (fgets (buffer, 120, pFile) != NULL) {
    if (known_macs == NULL) {
      known_macs = (macdb *) malloc (sizeof(macdb));
      known_macs->next = NULL;
      curr_node = known_macs;
    } else {
      curr_node->next = (macdb *) malloc (sizeof(macdb));
      curr_node = curr_node->next;
      curr_node->next = NULL;
    }
    mac = curr_node->mac;
    vendor = curr_node->vendor;
    memcpy (mac, buffer, 17); // not sure about 17 (mac address characters)
    mac[17] = '\0';
    for (i=0; i < 8; i++) {
      if (mac[i] == '-') mac[i] = ':';
    }
    memcpy (vendor, buffer+18, 80);
    for (i=0; i < 80; i++) {
      if (vendor[i] == '\n') vendor[i] = '\0';
    }
//    printf ("Added %s - %s\n", mac, vendor);
  }
}


// Reads a CSV file (fileName)
// Inserts the APs into a linked list of APs (firstAp)
// Inserts the Enddevs into a linked list of Enddevs (firstEnddev)
// Returns a devset with the addresses of the first AP and Devset
// because the values passed in will be NULL if this is the first file read
devset readCSVFile (char * fileName, ap *firstAp, enddev *firstEnddev, const int lastFile) {
  FILE *pFile;
  long i=0, j=0, k=0;
  long lSize;
  char *buffer;
  char mac[9];
  char power[5];
  char currWord[80];
  char description[16][80];
  char first_time_seen[80];
  int keepDate;
  size_t result;
  ap *currAp = NULL;
  ap *lastAp = firstAp;
  ap *tempAp = NULL;
  enddev *currEnddev = NULL;
  enddev *lastEnddev = firstEnddev;
  enddev *tempEnddev = NULL;
  devset dset;

  pFile = fopen (fileName, "r");

  if (pFile == NULL) {
   printf ("readCSVFile - Error opening file: %s\n", fileName);
   exit(1);
  }

  if (lastAp != NULL) while (lastAp->next != NULL) lastAp = lastAp->next;
  if (lastEnddev != NULL) while (lastEnddev->next != NULL) lastEnddev = lastEnddev->next;
  
  // obtain file size
  fseek (pFile, 0, SEEK_END);
  lSize = ftell (pFile);
  rewind (pFile);

  // allocate memory to contain the whole file
  buffer = (char*) malloc(sizeof(char) * lSize);
  if (buffer == NULL) {
    fputs ("Memory error\n", stderr);
    exit(2);
  }

  // copy the file into the buffer
  result = fread (buffer, 1, lSize, pFile);
  if (result != lSize) {
    fputs ("Reading error\n", stderr);
    exit(3);
  }
 
  // Skip the first two lines
  j=0;
  for (i=0; i < lSize-1; i++) {
    if (buffer[i] == '\n') j++;
    if (j == 2) {
      i++;
      break;
    }
  }

  // Read the list of aps
  while (i < lSize-1) {
    //Check if we are at the end of the ap list
    if (buffer[i] == '\n') {
      i++;
      break;
    } else if (buffer[i]== '\r' && buffer[i+1] == '\n') {
      i += 2;
      break;
    }
    // Read the next AP
    i = getWord (currWord, buffer, i, lSize);
    keepDate = 0;
    if (firstAp == NULL) {
      firstAp = (ap *) malloc (sizeof(ap));
      firstAp->next = NULL;
      firstAp->new = lastFile ? 1 : 0;
      firstAp->old = 0;
      firstAp->maxPwrLevel = -100;
      bzero(firstAp->maxPwrTime, 80);
      firstAp->lat = firstAp->lon = 0.0;
      lastAp = currAp = firstAp;
    } else {
      currAp = findApByBSSID (firstAp, currWord);
      if (currAp == NULL) {
        currAp = (ap *) malloc (sizeof(ap));
        currAp->next = NULL;
        currAp->new = lastFile ? 1 : 0;
        currAp->old = 0;
        currAp->maxPwrLevel = -100;
        bzero(currAp->maxPwrTime, 80);
        currAp->lat = currAp->lon = 0.0;
        lastAp->next = currAp;
        lastAp = currAp;
      } else {
        currAp->new = 0;
        currAp->old = lastFile ? 1 : 0;
        strcpy(first_time_seen, currAp->first_time_seen);
        keepDate = 1;
      }
    }

    strcpy (currAp->bssid, currWord);
    memcpy (mac, currAp->bssid, 8);
    mac[8] = '\0';
    strcpy (currAp->vendor, findVendorByMAC (mac_database, mac));
    // "Vendor" is actually the description in this case
    strcpy (currAp->desc, findVendorByMAC (known_macs, currAp->bssid));
    i = getWord (currAp->first_time_seen, buffer, i, lSize);
    if (keepDate) strcpy(currAp->first_time_seen, first_time_seen);
    i = getWord (currAp->last_time_seen, buffer, i, lSize);
    i = getWord (currAp->channel, buffer, i, lSize);
    i = getWord (currAp->speed, buffer, i, lSize);
    i = getWord (currAp->privacy, buffer, i, lSize);
    i = getWord (currAp->cipher, buffer, i, lSize);
    i = getWord (currAp->authentication, buffer, i, lSize);
    i = getWord (power, buffer, i, lSize);
    currAp->oldPower = 0;
    if (currAp->old) currAp->oldPower = currAp->power;
    currAp->power = atoi(power);
    i = getWord (currAp->beacons, buffer, i, lSize);
    i = getWord (currAp->ivs, buffer, i, lSize);
    i = getWord (currAp->lan_ip, buffer, i, lSize);
    i = getWord (currAp->id_length, buffer, i, lSize);
    i = getEssid (currAp->essid, buffer, i, lSize);
    i = getWord (currAp->key, buffer, i, lSize);
    strcpy (currAp->fileName, fileName);
    // I dont actually use the time1 and time2 for anything yet.
    sscanf (currAp->first_time_seen, "%d-%d-%d %d:%d:%d", 
      &(currAp->time1.year),
      &(currAp->time1.month),
      &(currAp->time1.day),
      &(currAp->time1.hour),
      &(currAp->time1.minute),
      &(currAp->time1.second));
    sscanf (currAp->last_time_seen, "%d-%d-%d %d:%d:%d",
      &(currAp->time2.year),
      &(currAp->time2.month),
      &(currAp->time2.day),
      &(currAp->time2.hour),
      &(currAp->time2.minute),
      &(currAp->time2.second));
    if (currAp->power > currAp->maxPwrLevel && currAp->power < -1) {
      currAp->maxPwrLevel = currAp->power;
      strcpy(currAp->maxPwrTime, currAp->last_time_seen);
    }
    if (strcmp(currAp->maxPwrTime, "") == 0) strcpy(currAp->maxPwrTime, currAp->last_time_seen);
//    printf("added %s\n", currAp->bssid);
  }

  // Skip the description line
  while (i < lSize) {
    if (buffer[i] == '\n') {
      i++; break;
    }
    i++;
  }

  // Read the end devices
  while (i < lSize) {
    //Check if we are at the end of the end device list
    if (buffer[i] == '\n') {
      i++;
      break;
    } else if (buffer[i]== '\r' && buffer[i+1] == '\n') {
      i += 2;
      break;
    }
    i = getWord (currWord, buffer, i, lSize);
    keepDate = 0;
    if (firstEnddev == NULL) {
      firstEnddev = (enddev *) malloc (sizeof(enddev));
      firstEnddev->next = NULL;
      firstEnddev->new = lastFile ? 1 : 0;
      firstEnddev->old = 0;
      firstEnddev->maxPwrLevel = -100;
      firstEnddev->lat = firstEnddev->lon = 0.0;
      bzero(firstEnddev->maxPwrTime, 80);
      lastEnddev = currEnddev = firstEnddev;
    } else {
      currEnddev = findEnddevByMAC (firstEnddev, currWord);
      if (currEnddev == NULL) {
        currEnddev = (enddev *) malloc (sizeof(enddev));
        currEnddev->next = NULL;
        currEnddev->new = lastFile ? 1 : 0;
        currEnddev->old = 0;
        currEnddev->maxPwrLevel = -100;
        bzero(currEnddev->maxPwrTime, 80);
        currEnddev->lat = currEnddev->lon = 0.0;
        lastEnddev->next = currEnddev;
        lastEnddev = currEnddev;
      } else {
        strcpy (first_time_seen, currEnddev->first_time_seen);
        currEnddev->new = 0;
        currEnddev->old = lastFile ? 1 : 0;
        keepDate = 1;
      }
    }
    strcpy (currEnddev->station_mac, currWord);
    memcpy (mac, currEnddev->station_mac, 8);
    mac[8] = '\0';
    strcpy (currEnddev->vendor, findVendorByMAC (mac_database, mac));
    // "Vendor" is actually the description in this case
    strcpy (currEnddev->desc, findVendorByMAC (known_macs, currEnddev->station_mac));
    i = getWord (currEnddev->first_time_seen, buffer, i, lSize);
    if (keepDate) strcpy (currEnddev->first_time_seen, first_time_seen);
    i = getWord (currEnddev->last_time_seen, buffer, i, lSize);
    i = getWord (power, buffer, i, lSize);
    currEnddev->oldPower = 0;
    if (currEnddev->old) currEnddev->oldPower = currEnddev->power;
    currEnddev->power = atoi(power);
    i = getWord (currEnddev->packets, buffer, i, lSize);
    i = getWord (currEnddev->bssid, buffer, i, lSize);
    if (strcmp (currEnddev->bssid, "(not associated)") != 0) {
      currAp = findApByBSSID (firstAp, currEnddev->bssid);
      if (currAp != NULL) {
        strcpy(currEnddev->essid, currAp->essid);
      } else {
        strcpy(currEnddev->essid, "");
      }
    } else {
      strcpy(currEnddev->essid, "");
    }
    strcpy (currEnddev->fileName, fileName);
    sscanf (currEnddev->first_time_seen, "%d-%d-%d %d:%d:%d",
      &(currEnddev->time1.year),
      &(currEnddev->time1.month),
      &(currEnddev->time1.day),
      &(currEnddev->time1.hour),
      &(currEnddev->time1.minute),
      &(currEnddev->time1.second));
    sscanf (currEnddev->last_time_seen, "%d-%d-%d %d:%d:%d",
      &(currEnddev->time2.year),
      &(currEnddev->time2.month),
      &(currEnddev->time2.day),
      &(currEnddev->time2.hour),
      &(currEnddev->time2.minute),
      &(currEnddev->time2.second));
    if (currEnddev->power > currEnddev->maxPwrLevel && currEnddev->power < -1) {
      currEnddev->maxPwrLevel = currEnddev->power;
      strcpy(currEnddev->maxPwrTime, currEnddev->last_time_seen);
    }
    if (strcmp(currEnddev->maxPwrTime, "") == 0) strcpy(currEnddev->maxPwrTime, currEnddev->last_time_seen);
    // Skip leading whitespace
    while (i < lSize && (buffer[i] == ' ' || buffer[i] == '\t')) i++;

    // Grab the whole list of ESSIDs
    j = 0;
    while (i < lSize-1 && j < 253) {
      if (buffer[i] == '\r' && buffer[i+1] == '\n') {
        i += 2;
        break;
      }
      if (buffer[i] == '\n') {
        i++;
        break;
      }
      currEnddev->probed_essids[j] = buffer[i];
      i++; j++;
    }
    currEnddev->probed_essids[j] = '\0';
  }

  fclose(pFile);
  free (buffer);

//  printf("Finished reading %s\n", fileName);  // Debug
  dset.s = firstAp;
  dset.e = firstEnddev;
  return dset;
}

int main (int argc, char **argv) {
  int i, lastFile, showAPs, showEnddevs;
  ap *firstAp = NULL;
  enddev *firstEnddev = NULL;
  gps *gps1 = NULL;
  devset dset;
/*
  FILE *csvFile = NULL;
  FILE *textFile = NULL;
  FILE *htmlFile = NULL;
 */
  FILE *tmpFile = NULL;
  char *fileToMonitor = NULL;
  char *filePrefix = NULL;
  char *gpsFile = NULL;
  char buffer[256];
//  int continuous = 0; //boolean

  // Set the default values
  onlyAddCommon = 0;
  onlyAddNew = 0;
  onlyAddOld = 0;
  deltaSpecified = 0;
  numInputFiles = 0;
  showAPs = 1;
  showEnddevs = 1;
  minPower = -100;
  maxPower = 0;
  minPowerDelta = -1;
  text_brief = 0;
  verbosity = 0;

  if (argc < 2) {
    printf ("Usage: %s [options] -w prefix file1 [file2] [file3]...[-l] [file n]\n", argv[0]);
    printf ("-a only show APs\n");
    printf ("-b print brief text in text mode\n");
    printf ("-e only show end devices\n");
//    printf ("-c specifies a csv file to output to\n");
//    printf ("-t specifies a text file to output to\n");
    printf ("-to prints text to stdout (cannot be used with -t)\n");
//    printf ("-h specifies a html file to output to\n");
    printf ("-g [file] specifies a GPS input file\n");
    printf ("-l specifies the last file (must be the last file specified)\n");
    printf ("-k [file] specifies a CSV file of known MAC addresses\n");
    printf ("-m only show APs and Stations in the file specified with -k\n");
    printf ("-d [delta] only shows devices whose power is stronger than before by [delta]\n");
    printf ("-n only shows APs and Stations that are new in the last file\n");
    printf ("-o only shows APs and Stations that are not new in the last file\n");
    printf ("-p [power] only shows APs and Stations with power greater than [power]\n");
    printf ("-P [power] only shows APs and Stations with power less than [power]\n");
    printf ("-v verbose output\n");
    printf ("-vv very verbose output\n");
    printf ("-w [prefix] specifies output file prefix\n");
    return 1;
  }

  firstAp = NULL;
  firstEnddev = NULL;

  tmpFile = fopen("/usr/share/aircrack-ng/airodump-ng-oui.txt", "r");
  if (tmpFile) {
    fclose(tmpFile);
    readMacDB ("/usr/share/aircrack-ng/airodump-ng-oui.txt");
  } else if (tmpFile = fopen("/etc/aircrack-ng/airodump-ng-oui.txt", "r")) {
    fclose(tmpFile);
    readMacDB ("/etc/aircrack-ng/airodump-ng-oui.txt");
  }
  for (i = 1; i < argc; i++) {
/*
    if (strcmp(argv[i], "-c") == 0 || strcmp(argv[i], "-C") == 0) {
      i++;
      if (i >= argc) {
        printf ("-c requires that you specify an output csv file.\n");
        exit(1);
      }
      csvFile = fopen(argv[i], "w");
      if (csvFile == NULL) {
        printf ("Error opening CSV file: %s\n", argv[i]);
        exit(1);
      }
      continue;
    }
*/
/*
    if (strcmp(argv[i], "-t") == 0 || strcmp(argv[i], "-T") == 0) {
      i++;
      if (i >= argc) {
        printf ("-t requires that you specify an output text file.\n");
        exit(1);
      }
      textFile = fopen(argv[i], "w");
      if (textFile == NULL) {
        printf ("Error opening text file: %s\n", argv[i]);
        exit(1);
      }
      continue;
    }
*/
    if (strcmp(argv[i], "-to") == 0 || strcmp(argv[i], "-TO") == 0) {
      if (textFile != NULL) {
        printf ("Error: -to cannot be used with -t\n");
        exit(1);
      }
      textFile = stdout;
      continue;
    }
/*
    if (strcmp(argv[i], "-h") == 0 || strcmp(argv[i], "-H") == 0) {
      i++;
      if (i >= argc) {
        printf ("-h requires that you specify an output text file.\n");
        exit(1);
      }
      htmlFile = fopen(argv[i], "w");
      if (htmlFile == NULL) {
        printf ("Error opening html file: %s\n", argv[i]);
        exit(1);
      }
      continue;
    }
*/
    if (strcmp(argv[i], "-a") == 0 || strcmp(argv[i], "-A") == 0) {
      showEnddevs = 0;
      continue;
    }
    if (strcmp(argv[i], "-b") == 0 || strcmp(argv[i], "-B") == 0) {
      text_brief = 1;
      continue;
    }
    if (strcmp(argv[i], "-d") == 0) {
      i++;
      if (i >= argc) {
        printf ("-d requires that you specify a minimum delta power level.\n");
        exit(1);
      }
      deltaSpecified = 1;
      minPowerDelta = atoi(argv[i]);
      continue;
    }
    if (strcmp(argv[i], "-e") == 0 || strcmp(argv[i], "-E") == 0) {
      showAPs = 0;
      continue;
    }
    if (strcmp(argv[i], "-g") == 0) {
      i++;
      if (i >= argc) {
        printf ("-g requires that you specify a GPS file.\n");
        continue;
      }
      gpsFile = argv[i];
      continue;
    }
    if (strcmp(argv[i], "-k") == 0 || strcmp(argv[i], "-K") == 0) {
      i++;
      if (i >= argc) {
        printf ("-k requires that you specify a MAC address file.\n");
        exit(1);
      }
      readKnownMacs(argv[i]);

      continue;
    }
    if (strcmp(argv[i], "-m") == 0 || strcmp(argv[i], "-M") == 0) {
      onlyShowKnown = 1;
      continue;
    }
    if (strcmp(argv[i], "-n") == 0 || strcmp(argv[i], "-N") == 0) {
      onlyAddNew = 1;
      continue;
    }
    if (strcmp(argv[i], "-o") == 0 || strcmp(argv[i], "-O") == 0) {
      onlyAddOld = 1;
      continue;
    }
    if (strcmp(argv[i], "-p") == 0) {
      i++;
      if (i >= argc) {
        printf ("-p requires that you specify a minimum power level.\n");
        exit(1);
      }
      minPower = atoi(argv[i]);
      continue;
    }
    if (strcmp(argv[i], "-P") == 0) {
      i++;
      if (i >= argc) {
        printf ("-P requires that you specify a maximum power level.\n");
        exit(1);
      }
      maxPower = atoi(argv[i]);
      continue;
    }
    if (strcmp(argv[i], "-v") == 0) {
      verbosity = 1;
      continue;
    }
    if (strcmp(argv[i], "-vv") == 0) {
      verbosity = 2;
      continue;
    }
    if (strcmp(argv[i], "-w") == 0) {
      i++;
      if (i >= argc) {
        printf ("-w requires that you specify an output prefix.\n");
        continue;
      }
      filePrefix = argv[i];
      continue;
    }
    // Note the lack of a continue statement after -l - this option must be last
    lastFile = 0;
    if (strcmp(argv[i], "-l") == 0 || strcmp(argv[i], "-L") == 0) {
      lastFile = 1;
      i++;
      if (i >= argc) {
        printf ("-l requires that you specify an input csv file.\n");
        exit(1);
      }
      fileToMonitor = argv[i];
    }
    if (verbosity) printf ("Reading CSV file: %s\n", argv[i]);
    dset = readCSVFile (argv[i], firstAp, firstEnddev, lastFile);
    firstAp = dset.s;
    firstEnddev = dset.e;
    numInputFiles++;
    if (verbosity >= 2) printf ("Finished reading CSV file.\n");
  }

  // Check for show stoppers
  if (numInputFiles == 0) {
    printf ("Error: no input files specified.\n");
    exit(1);
  }
  if (filePrefix == NULL) {
    printf ("Please specify an output prefix (-w option).\n");
    exit(1);
  }

  // Print the power level files
  strcpy (buffer, "");
  strcat (buffer, filePrefix);
  strcat (buffer, "-appower.csv");
  if (verbosity) printf ("Opening file: %s\n", buffer);
  tmpFile = fopen(buffer, "r");
  if (tmpFile) {
    readAPPowerFromFile (firstAp, tmpFile);
    fclose (tmpFile);
  }
  tmpFile = fopen(buffer, "w");
  printAPPowerToFileRec (firstAp, tmpFile);
  fclose(tmpFile);

  strcpy (buffer, "");
  strcat (buffer, filePrefix);
  strcat (buffer, "-stapower.csv");
  if (verbosity) printf ("Opening file: %s\n", buffer);
  tmpFile = fopen(buffer, "r");
  if (tmpFile) {
    readEnddevPowerFromFile (firstEnddev, tmpFile);
    fclose (tmpFile);
  }
  tmpFile = fopen(buffer, "w");
  printEndDevicesPowerToFileRec (firstEnddev, tmpFile);
  fclose(tmpFile);

  tmpFile = fopen(gpsFile, "r");
  if (tmpFile) {
    gps1 = readGPSFile(firstAp, firstEnddev, tmpFile);
    addGPSInfo (firstAp, firstEnddev, gps1);
    free_gps(gps1);
  }

//  csvFile = htmlFile = kmlFile = NULL;

  if (textFile != stdout) {
    strcpy (buffer, "");
    strcat (buffer, filePrefix);
    strcat (buffer, ".txt");
    textFile = fopen(buffer, "w");
  }

  strcpy (buffer, "");
  strcat (buffer, filePrefix);
  strcat (buffer, ".csv");
  csvFile = fopen(buffer, "w");

  strcpy (buffer, "");
  strcat (buffer, filePrefix);
  strcat (buffer, ".html");
  htmlFile = fopen(buffer, "w");

  if (gpsFile) {
    strcpy (buffer, "");
    strcat (buffer, filePrefix);
    strcat (buffer, ".kml");
    kmlFile = fopen (buffer, "w");
  }

/*
  if (textFile != NULL) {
    if (showAPs) printAPsToFileText (firstAp, textFile);
    if (showEnddevs) printEndDevicesToFileText (firstEnddev, textFile);
    if (textFile != stdout) fclose (textFile);
  }
*/
  if (csvFile && htmlFile && textFile) {
    if (kmlFile) fprintf (kmlFile, "<?xml version=\"1.0\" encoding=\"UTF-8\"?>\r\n<kml xmlns=\"http://www.opengis.net/kml/2.2\">\r\n<Document>\r\n");
    if (showAPs) {
      fprintf (csvFile, "BSSID, First time seen, Last time seen, channel, Speed, Privacy, Cipher, Authentication, "
      "Power, # beacons, # IV, LAN IP, ID-Length, ESSID, Key%s", CRLF);
      fprintf (htmlFile, "<html>%s<head></head><body>%s<table border=\"1\">%s", CRLF, CRLF, CRLF);
      fprintf (htmlFile, "<tr><td>BSSID</td><td>Vendor</td><td>First time seen</td><td>Last time seen</td><td>channel</td><td>Speed</td><td>Privacy</td><td>Cipher</td><td>Authentication</td>"
        "<td>Power</td><td># beacons</td><td># IV</td><td>LAN IP</td><td>ID-Length</td><td>ESSID</td><td>Key</td><td>Description</td></tr>%s", CRLF);

      printAPsToFileRec (firstAp);
    }
    if (showEnddevs) {
      fprintf (csvFile, "Station MAC, First time seen, Last time seen, Power, # packets, BSSID, Probed ESSIDs%s", CRLF);
      fprintf (htmlFile, "</table><table border=\"1\"><tr><td>Station MAC</td><td>Vendor</td><td>First time seen</td><td>Last time seen</td><td>Power</td><td># packets</td><td>BSSID</td>"
        "<td>ESSID</td><td>Probes</td><td>Description</td></tr>%s", CRLF);
      printEndDevicesToFileRec (firstEnddev);
    }
    fprintf (htmlFile, "</table>");
    if (kmlFile) {
      fprintf (kmlFile, "</Document>\r\n</kml>\r\n");
      fclose (kmlFile);
    }
    fclose (csvFile);
    fclose (htmlFile);
    fclose (textFile);
  }
/*
  if (htmlFile != NULL) {
    if (showAPs) {
      fprintf (htmlFile, "<html>%s<head></head><body>%s<table border=\"1\">%s", CRLF, CRLF, CRLF);
      fprintf (htmlFile, "<tr><td>BSSID</td><td>Vendor</td><td>First time seen</td><td>Last time seen</td><td>channel</td><td>Speed</td><td>Privacy</td><td>Cipher</td><td>Authentication</td>", CRLF);
      fprintf (htmlFile, "<td>Power</td><td># beacons</td><td># IV</td><td>LAN IP</td><td>ID-Length</td><td>ESSID</td><td>Key</td></tr>%s", CRLF);
      printAPsToFileHTML (firstAp, htmlFile);
      fprintf (htmlFile, "</table>%s", CRLF);
    }
    if (showEnddevs) {
      fprintf (htmlFile, "<table border=\"1\">%s", CRLF);
      fprintf (htmlFile, "<tr><td>Station MAC</td><td>Vendor</td><td>First time seen</td><td>Last time seen</td><td>Power</td><td># packets</td><td>BSSID</td><td>ESSID</td><td>Probes</td></tr>%s", CRLF);
      printEndDevicesToFileHTML (firstEnddev, htmlFile);
      fprintf (htmlFile, "</table>%s</body>%s</html>", CRLF, CRLF, CRLF);
    }
    fclose (htmlFile);
  }
*/
/*
  if (continuous && fileToMonitor == NULL) {
    printf ("Error, last file not specified.\n");
    return 1;
  }
  while (continuous) {
    dset = readCSVFile (fileToMonitor, firstAp, firstEnddev, lastFile);
    firstAp = dset.s;
    firstEnddev = dset.e;
    if (showAPs) printAPsToFileText (firstAp, stdout);
    if (showEnddevs) printEndDevicesToFileText (firstEnddev, stdout);
    sleep(3);
  }
*/
  free_ap(firstAp);
  free_enddev(firstEnddev);

  return 0;
}

