/*
    Airodump CSV Tools v0.2
    Merges and parses Airodump CSV files and outputs them to CSV, HTML, or text.
    Copyright (C) 2014 Christopher Bolduc

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#define CRLF "\r\n"

// Linked list of Access Points
typedef struct ap {
  char bssid[80];
  char vendor[80];
  char first_time_seen[80];
  char last_time_seen[80];
  char channel[80];
  char speed[80];
  char privacy[80];
  char cipher[80];
  char authentication[80];
  char power[80];
  char beacons[80];
  char ivs[80];
  char lan_ip[80];
  char id_length[80];
  char essid[80];
  char key[80];
  char fileName[80];

  int new;
  int old;
  struct ap *next;
} ap;

// Linked list of End Devices
typedef struct enddev {
  char station_mac[80];
  char vendor[80];
  char first_time_seen[80];
  char last_time_seen[80];
  char power[80];
  char packets[80];
  char bssid[80];
  char essid[80];
  char probed_essids[255];
  char fileName[80];

  int new;
  int old;
  struct enddev *next;
} enddev;

typedef struct macdb {
  char mac[10];
  char vendor[80];

  struct macdb *next;
} macdb;

typedef struct devset {
  ap *s;
  enddev *e;
} devset;

// Prototypes
void free_ap (ap *s);
void free_enddev (enddev *e);
long getWord(char *currWord, char *buffer, long i, long lSize);
ap *findApByBSSID (ap *s, char *key);
char *findVendorByMAC (macdb * m, char * key);
devset readCSVFile (char * fileName, ap *firstAp, enddev *firstEnddev, const int lastFile);
void readMacDB (char * fileName);
void printAPToFileText (ap *a, FILE *f);
void printAPsToFileText (ap *a, FILE *f);
void printEndDeviceToFileText (enddev *e, FILE *f);
void printEndDevicesToFileText (enddev *e, FILE *f);
void printAPToFileCSV (ap *a, FILE *f);
void printAPsToFileCSV (ap *a, FILE *f);
void printEndDeviceToFileCSV (enddev *e, FILE *f);
void printEndDevicesToFileCSV (enddev *e, FILE *f);
void printAPToFileHTML (ap *a, FILE *f);
void printAPsToFileHTML (ap *a, FILE *f);
void printEndDeviceToFileHTML (enddev *e, FILE *f);
void printEndDevicesToFileHTML (enddev *e, FILE *f);

// Globals
int onlyAddCommon; // boolean
int onlyAddNew; // boolean
int onlyAddOld; // boolean
int numInputFiles;
macdb *mac_database;

// Free the linked list of APs
void free_ap (ap *s) {
  if (s == NULL) return;
  free_ap(s->next);
  free(s);
}

// Free the linked list of Enddevs
void free_enddev (enddev *e) {
  if (e == NULL) return;
  free_enddev(e->next);
  free(e);
}

// Reads one word from the input file (buffer) and places it into currWord
// Returns the position in buffer
long getWord(char *currWord, char *buffer, long i, long lSize) {
  long j=0;

  // Skip leading whitespace
  while (i < lSize && (buffer[i] == ' ' || buffer[i] == '\t')) i++;

  while (i < lSize-1) {
    if (buffer[i] == '\r' && buffer[i+1] == '\n') {
      // Why, Microsoft?
      i += 2;
      currWord[j] = '\0';
//      printf ("getWord: %s\n", currWord);
      return i;
    }
    if (buffer[i] == ',' || buffer[i] == '\n') {
      currWord[j] = '\0';
//      printf ("getWord: %s\n", currWord);
      i++;
      return i;
    }
    currWord[j] = buffer[i];
    i++;
    j++;
  }
}

// Finds an AP in the linked list given the BSSID (key)
ap *findApByBSSID (ap *s, char *key) {
//  printf ("Got next: %d\n", temp);
//  printf ("Next BSSID: %s\n", s->next->bssid);
  if (strcmp (s->bssid, key) == 0) return s;
  if (s->next == NULL) return NULL;
  return findApByBSSID(s->next, key);
}

char *findVendorByMAC (macdb * m, char * key) {
//  printf ("Searching for: %s\n", key);
  if (strcmp(m->mac, key) == 0) return m->vendor;
  if (m->next == NULL) return "";
  return findVendorByMAC(m->next, key);
}

// Finds an Enddev in the linked list given the Station MAC (key)
enddev *findEnddevByMAC (enddev *e, char *key) {
  if (strcmp (e->station_mac, key) == 0) return e;
  if (e->next == NULL) return NULL;
  return findEnddevByMAC(e->next, key);
}

void printAPAlert (ap *a) {
  printf ("BSSID %s seen at %s", a->bssid, a->last_time_seen);
}

// Prints a single AP (a) to a file (f)
void printAPToFileHTML (ap *a, FILE *f) {
  fprintf (f, "<tr><td>%s</td><td>%s</td><td>%s</td><td>%s</td><td>%s</td><td>%s</td><td>%s</td><td>%s</td><td>%s</td><td>%s</td><td>%s</td><td>%s</td><td>%s</td><td>%s</td><td>%s</td><td>%s</td></tr>%s",
    a->bssid, a->vendor, a->first_time_seen, a->last_time_seen, a->channel, a->speed, a->privacy, a->cipher,
    a->authentication, a->power, a->beacons, a->ivs, a->lan_ip, a->id_length, a->essid, a->key,
    CRLF);
}

// Prints all of the APs in the linked list (a) to a file (f)
void printAPsToFileHTML (ap *a, FILE *f) {
  int result;
  int i;
  ap *currAp = a;

  if (a == NULL) {
    fprintf(f, "%s", CRLF);
    return;
  }
  if (onlyAddNew) {
    if (a->new) printAPToFileHTML (a, f);
  } else if (onlyAddOld) {
    if (a->old) printAPToFileHTML(a, f);
  } else {
    printAPToFileHTML (a, f);
  }

  result = ferror (f);
  if (result) {
    printf ("printAPsToFileHTML fprintf returned error: %d\n", result);
    return;
  }
  printAPsToFileHTML (a->next, f);
}

// Prints a single Enddev (e) to a file (f)
void printEndDeviceToFileHTML (enddev *e, FILE *f) {
  fprintf (f, "<tr><td>%s</td><td>%s</td><td>%s</td><td>%s</td><td>%s</td><td>%s</td><td>%s</td><td>%s</td><td>%s</td></tr>%s", e->station_mac, e->vendor, e->first_time_seen,
    e->last_time_seen, e->power, e->packets, e->bssid, e->essid, e->probed_essids, CRLF);
}

// Prints a linked list of Enddevs (e) to a file (f)
void printEndDevicesToFileHTML (enddev *e, FILE *f) {
  int result;
  int i;
  enddev *curr = e;

  if (e == NULL) {
    fprintf(f, "%s", CRLF);
    return;
  }
  if (onlyAddNew) {
    if (e->new) printEndDeviceToFileHTML (e, f);
  } else if (onlyAddOld) {
    if (e->old) printEndDeviceToFileHTML (e, f);
  } else {
    printEndDeviceToFileHTML (e, f);
  }

  result = ferror (f);
  if (result) {
    printf ("printEndDevicesToFileCSV fprintf returned error: %d\n", result);
    return;
  }
  printEndDevicesToFileHTML (e->next, f);
}


// Prints a single AP (a) to a file (f)
void printAPToFileCSV (ap *a, FILE *f) {
  fprintf (f, "%s, %s, %s, %s, %s, %s, %s, %s, %s, %s, %s, %s, %s, %s, %s%s",
    a->bssid, a->first_time_seen, a->last_time_seen, a->channel, a->speed, a->privacy, a->cipher,
    a->authentication, a->power, a->beacons, a->ivs, a->lan_ip, a->id_length, a->essid, a->key, CRLF);
}

// Prints all of the APs in the linked list (a) to a file (f)
void printAPsToFileCSV (ap *a, FILE *f) {
  int result;
  int i;
  ap *currAp = a;

  if (a == NULL) {
    fprintf(f, "%s", CRLF);
    return;
  }
  if (onlyAddNew) {
    if (a->new) printAPToFileCSV(a, f);
  } else if (onlyAddOld) {
    if (a->old) printAPToFileCSV(a, f);
  } else {
    printAPToFileCSV (a, f);
  }

  result = ferror (f);
  if (result) {
    printf ("printAPsToFileCSV fprintf returned error: %d\n", result);
    return;
  }
  printAPsToFileCSV (a->next, f);
}

// Prints a single Enddev (e) to a file (f)
void printEndDeviceToFileCSV (enddev *e, FILE *f) {
  fprintf (f, "%s, %s, %s, %s, %s, %s, %s%s", e->station_mac, e->first_time_seen,
    e->last_time_seen, e->power, e->packets, e->bssid, e->probed_essids, CRLF);
}

// Prints a linked list of Enddevs (e) to a file (f)
void printEndDevicesToFileCSV (enddev *e, FILE *f) {
  int result;
  int i;
  enddev *curr = e;

  if (e == NULL) {
    fprintf(f, "%s", CRLF);
    return;
  }
  if (onlyAddNew) {
    if (e->new) printEndDeviceToFileCSV (e, f);
  } else if (onlyAddOld) {
    if (e->old) printEndDeviceToFileCSV (e, f);
  } else {
    printEndDeviceToFileCSV (e, f);
  }

  result = ferror (f);
  if (result) {
    printf ("printEndDevicesToFileCSV fprintf returned error: %d\n", result);
    return;
  }
  printEndDevicesToFileCSV (e->next, f);
}

// Prints a single ap (a) to a file (f)
void printAPToFileText (ap *a, FILE *f) {
  fprintf (f, "BSSID: %s%s", a->bssid, CRLF);
  fprintf (f, "Vendor: %s%s", a->vendor, CRLF);
  fprintf (f, "First time seen: %s%s", a->first_time_seen, CRLF);
  fprintf (f, "Last time seen: %s%s", a->last_time_seen, CRLF);
  fprintf (f, "Channel: %s%s", a->channel, CRLF);
  fprintf (f, "Speed: %s%s", a->speed, CRLF);
  fprintf (f, "Privacy: %s%s", a->privacy, CRLF);
  fprintf (f, "Cipher: %s%s", a->cipher, CRLF);
  fprintf (f, "Authentication: %s%s", a->authentication, CRLF);
  fprintf (f, "Power: %s%s", a->power, CRLF);
  fprintf (f, "Beacons: %s%s", a->beacons, CRLF);
  fprintf (f, "IVs: %s%s", a->ivs, CRLF);
  fprintf (f, "LAN IP: %s%s", a->lan_ip, CRLF);
  fprintf (f, "ID-length: %s%s", a->id_length, CRLF);
  fprintf (f, "ESSID: %s%s", a->essid, CRLF);
  fprintf (f, "Key: %s%s", a->key, CRLF);
  fprintf (f, "File: %s%s%s", a->fileName, CRLF, CRLF);
}

// Prints a linked list of APs (a) to a file (f)
void printAPsToFileText (ap *a, FILE *f) {
  int i, result;
  ap *currAp = a;

  if (a == NULL) return;
  if (onlyAddNew) {
    if (a->new) printAPToFileText (a, f);
  } else if (onlyAddOld) {
    if (a->old) printAPToFileText (a, f);
  } else {
    printAPToFileText (a, f);
  }

  result = ferror (f);
  if (result) {
    printf ("printAPsToFileText fprintf returned error: %d\n", result);
    return;
  }
  printAPsToFileText (a->next, f);
}

// Prints a single Enddev (e) to a file (f)
void printEndDeviceToFileText (enddev *e, FILE *f) {
  fprintf (f, "Station MAC: %s%s", e->station_mac, CRLF);
  fprintf (f, "Vendor: %s%s", e->vendor, CRLF);
  fprintf (f, "First time seen: %s%s", e->first_time_seen, CRLF);
  fprintf (f, "Last time seen: %s%s", e->last_time_seen, CRLF);
  fprintf (f, "Power: %s%s", e->power, CRLF);
  fprintf (f, "Packet count: %s%s", e->packets, CRLF);
  fprintf (f, "BSSID: %s%s", e->bssid, CRLF);
  fprintf (f, "ESSID: %s%s", e->essid, CRLF);
  fprintf (f, "Probed ESSIDs: %s%s", e->probed_essids, CRLF);
  fprintf (f, "File %s%s%s", e->fileName, CRLF, CRLF);
}

// Prints a linked list of Enddevs (e) to a file (f)
void printEndDevicesToFileText (enddev *e, FILE *f) {
  int i, result;
  enddev *curr = e;

  if (e == NULL) return;
  if (onlyAddNew) {
    if (e->new) printEndDeviceToFileText (e, f);
  } else if (onlyAddOld) {
    if (e->old) printEndDeviceToFileText (e, f);
  } else {
    printEndDeviceToFileText (e, f);
  }

  result = ferror (f);
  if (result) {
    printf ("printEndDevicesToFileText fprintf returned error: %d\n", result);
    return;
  }
  printEndDevicesToFileText (e->next, f);
}

void readMacDB (char * fileName) {
  FILE *pFile;
  long lSize;
  char buffer[120];
  char *mac, *vendor;
  macdb *curr_node;
  int i;

  pFile = fopen (fileName, "r");

  if (pFile == NULL) {
   printf ("Error opening file: %s\n", fileName);
   exit(1);
  }

  while ( fgets (buffer, 120, pFile) != NULL) {
    if (mac_database == NULL) {
      mac_database = (macdb *) malloc (sizeof(macdb));
      mac_database->next = NULL;
      curr_node = mac_database;
    } else {
      curr_node->next = (macdb *) malloc (sizeof(macdb));
      curr_node = curr_node->next;
      curr_node->next = NULL;
    }
    mac = curr_node->mac;
    vendor = curr_node->vendor;
    memcpy (mac, buffer, 8);
    mac[8] = '\0';
    for (i=0; i < 8; i++) {
      if (mac[i] == '-') mac[i] = ':';
    }
    memcpy (vendor, buffer + 18, 80);
    for (i=0; i < 80; i++) {
      if (vendor[i] == '\n') vendor[i] = '\0';
    }
  }
}

// Reads a CSV file (fileName)
// Inserts the APs into a linked list of APs (firstAp)
// Inserts the Enddevs into a linked list of Enddevs (firstEnddev)
// Returns a devset with the addresses of the first AP and Devset
// because the values passed in will be NULL if this is the first file read
devset readCSVFile (char * fileName, ap *firstAp, enddev *firstEnddev, const int lastFile) {
  FILE *pFile;
  long i=0, j=0, k=0;
  long lSize;
  char *buffer;
  char mac[9];
  char currWord[80];
  char description[16][80];
  char first_time_seen[80];
  int keepDate;
  size_t result;
  ap *currAp = NULL;
  ap *lastAp = firstAp;
  ap *tempAp = NULL;
  enddev *currEnddev = NULL;
  enddev *lastEnddev = firstEnddev;
  enddev *tempEnddev = NULL;
  devset dset;

  pFile = fopen (fileName, "r");

  if (pFile == NULL) {
   printf ("Error opening file: %s\n", fileName);
   exit(1);
  }

  if (lastAp != NULL) while (lastAp->next != NULL) lastAp = lastAp->next;
  if (lastEnddev != NULL) while (lastEnddev->next != NULL) lastEnddev = lastEnddev->next;
  
  // obtain file size
  fseek (pFile, 0, SEEK_END);
  lSize = ftell (pFile);
  rewind (pFile);

  // allocate memory to contain the whole file
  buffer = (char*) malloc(sizeof(char) * lSize);
  if (buffer == NULL) {
    fputs ("Memory error\n", stderr);
    exit(2);
  }

  // copy the file into the buffer
  result = fread (buffer, 1, lSize, pFile);
  if (result != lSize) {
    fputs ("Reading error\n", stderr);
    exit(3);
  }
 
  // Skip the first two lines
  j=0;
  for (i=0; i < lSize-1; i++) {
    if (buffer[i] == '\n') j++;
    if (j == 2) {
      i++;
      break;
    }
  }

  // Read the list of aps
  while (i < lSize-1) {
    //Check if we are at the end of the ap list
    if (buffer[i] == '\n') {
      i++;
      break;
    } else if (buffer[i]== '\r' && buffer[i+1] == '\n') {
      i += 2;
      break;
    }
    // Read the next AP
    i = getWord (currWord, buffer, i, lSize);
    keepDate = 0;
    if (firstAp == NULL) {
      firstAp = (ap *) malloc (sizeof(ap));
      firstAp->next = NULL;
      firstAp->new = lastFile ? 1 : 0;
      firstAp->old = 0;
      lastAp = currAp = firstAp;
    } else {
      currAp = findApByBSSID (firstAp, currWord);
      if (currAp == NULL) {
        currAp = (ap *) malloc (sizeof(ap));
        currAp->next = NULL;
        currAp->new = lastFile ? 1 : 0;
        currAp->old = 0;
        lastAp->next = currAp;
        lastAp = currAp;
      } else {
        currAp->new = 0;
        currAp->old = lastFile ? 1 : 0;
        strcpy(first_time_seen, currAp->first_time_seen);
        keepDate = 1;
      }
    }

    strcpy (currAp->bssid, currWord);
    memcpy (mac, currAp->bssid, 8);
    mac[8] = '\0';
    strcpy (currAp->vendor, findVendorByMAC (mac_database, mac));
    i = getWord (currAp->first_time_seen, buffer, i, lSize);
    if (keepDate) strcpy(currAp->first_time_seen, first_time_seen);
    i = getWord (currAp->last_time_seen, buffer, i, lSize);
    i = getWord (currAp->channel, buffer, i, lSize);
    i = getWord (currAp->speed, buffer, i, lSize);
    i = getWord (currAp->privacy, buffer, i, lSize);
    i = getWord (currAp->cipher, buffer, i, lSize);
    i = getWord (currAp->authentication, buffer, i, lSize);
    i = getWord (currAp->power, buffer, i, lSize);
    i = getWord (currAp->beacons, buffer, i, lSize);
    i = getWord (currAp->ivs, buffer, i, lSize);
    i = getWord (currAp->lan_ip, buffer, i, lSize);
    i = getWord (currAp->id_length, buffer, i, lSize);
    i = getWord (currAp->essid, buffer, i, lSize);
    i = getWord (currAp->key, buffer, i, lSize);
    strcpy (currAp->fileName, fileName);

//    printf("added %s\n", currAp->bssid);
  }

  // Skip the description line
  while (i < lSize) {
    if (buffer[i] == '\n') {
      i++; break;
    }
    i++;
  }

  // Read the end devices
  while (i < lSize) {
    //Check if we are at the end of the end device list
    if (buffer[i] == '\n') {
      i++;
      break;
    } else if (buffer[i]== '\r' && buffer[i+1] == '\n') {
      i += 2;
      break;
    }
    i = getWord (currWord, buffer, i, lSize);
    keepDate = 0;
    if (firstEnddev == NULL) {
      firstEnddev = (enddev *) malloc (sizeof(enddev));
      firstEnddev->next = NULL;
      firstEnddev->new = lastFile ? 1 : 0;
      firstEnddev->old = 0;
      lastEnddev = currEnddev = firstEnddev;
    } else {
      currEnddev = findEnddevByMAC (firstEnddev, currWord);
      if (currEnddev == NULL) {
        currEnddev = (enddev *) malloc (sizeof(enddev));
        currEnddev->next = NULL;
        currEnddev->new = lastFile ? 1 : 0;
        currEnddev->old = 0;
        lastEnddev->next = currEnddev;
        lastEnddev = currEnddev;
      } else {
        strcpy (first_time_seen, currEnddev->first_time_seen);
        currEnddev->new = 0;
        currEnddev->old = lastFile ? 1 : 0;
        keepDate = 1;
      }
    }
    strcpy (currEnddev->station_mac, currWord);
    memcpy (mac, currEnddev->station_mac, 8);
    mac[8] = '\0';
    strcpy (currEnddev->vendor, findVendorByMAC (mac_database, mac));
    i = getWord (currEnddev->first_time_seen, buffer, i, lSize);
    if (keepDate) strcpy (currEnddev->first_time_seen, first_time_seen);
    i = getWord (currEnddev->last_time_seen, buffer, i, lSize);
    i = getWord (currEnddev->power, buffer, i, lSize);
    i = getWord (currEnddev->packets, buffer, i, lSize);
    i = getWord (currEnddev->bssid, buffer, i, lSize);
    if (strcmp (currEnddev->bssid, "(not associated)") != 0) {
      currAp = findApByBSSID (firstAp, currEnddev->bssid);
      if (currAp != NULL) {
        strcpy(currEnddev->essid, currAp->essid);
      } else {
        strcpy(currEnddev->essid, "");
      }
    } else {
      strcpy(currEnddev->essid, "");
    }
    strcpy (currEnddev->fileName, fileName);

    // Skip leading whitespace
    while (i < lSize && (buffer[i] == ' ' || buffer[i] == '\t')) i++;

    // Grab the whole list of ESSIDs
    j = 0;
    while (i < lSize-1 && j < 253) {
      if (buffer[i] == '\r' && buffer[i+1] == '\n') {
        i += 2;
        break;
      }
      if (buffer[i] == '\n') {
        i++;
        break;
      }
      currEnddev->probed_essids[j] = buffer[i];
      i++; j++;
    }
    currEnddev->probed_essids[j] = '\0';
  }

  fclose(pFile);
  free (buffer);

//  printf("Finished reading %s\n", fileName);  // Debug
  dset.s = firstAp;
  dset.e = firstEnddev;
  return dset;
}

int main (int argc, char **argv) {
  int i, temp, showAPs, showEnddevs;
  ap *firstAp = NULL;
  enddev *firstEnddev = NULL;
  devset dset;
  FILE *csvFile = NULL;
  FILE *textFile = NULL;
  FILE *htmlFile = NULL;

  onlyAddCommon = 0;
  onlyAddNew = 0;
  onlyAddOld = 0;
  numInputFiles = 0;
  showAPs = 1;
  showEnddevs = 1;

  if (argc < 2) {
    printf ("Usage: %s [options] file1 [file2] [file3]...[-l] [file n]\n", argv[0]);
    printf ("-a only show APs\n");
    printf ("-e only show end devices\n");
    printf ("-c specifies a csv file to output to\n");
    printf ("-t specifies a text file to output to\n");
    printf ("-to prints text to stdout (cannot be used with -t)\n");
    printf ("-h specifies a html file to output to\n");
    printf ("-l specifies the last file (must be the last file specified)\n");
    printf ("-n only shows APs and Stations that are new in the last file\n");
    printf ("-o only shows APs and Stations that are not new in the last file\n");
    return 1;
  }

  firstAp = NULL;
  firstEnddev = NULL;

  readMacDB ("/etc/aircrack-ng/airodump-ng-oui.txt");
  for (i = 1; i < argc; i++) {
    if (strcmp(argv[i], "-c") == 0 || strcmp(argv[i], "-C") == 0) {
      i++;
      if (i >= argc) {
        printf ("-c requires that you specify an output csv file.\n");
        exit(1);
      }
      csvFile = fopen(argv[i], "w");
      if (csvFile == NULL) {
        printf ("Error opening CSV file: %s\n", argv[i]);
        exit(1);
      }
      continue;
    }
    if (strcmp(argv[i], "-t") == 0 || strcmp(argv[i], "-T") == 0) {
      i++;
      if (i >= argc) {
        printf ("-t requires that you specify an output text file.\n");
        exit(1);
      }
      textFile = fopen(argv[i], "w");
      if (textFile == NULL) {
        printf ("Error opening text file: %s\n", argv[i]);
        exit(1);
      }
      continue;
    }
    if (strcmp(argv[i], "-to") == 0 || strcmp(argv[i], "-TO") == 0) {
      if (textFile != NULL) {
        printf ("Error: -to cannot be used with -t\n");
        exit(1);
      }
      textFile = stdout;
      continue;
    }
    if (strcmp(argv[i], "-h") == 0 || strcmp(argv[i], "-H") == 0) {
      i++;
      if (i >= argc) {
        printf ("-h requires that you specify an output text file.\n");
        exit(1);
      }
      htmlFile = fopen(argv[i], "w");
      if (htmlFile == NULL) {
        printf ("Error opening html file: %s\n", argv[i]);
        exit(1);
      }
      continue;
    }
    if (strcmp(argv[i], "-a") == 0 || strcmp(argv[i], "-A") == 0) {
      showEnddevs = 0;
      continue;
    }
    if (strcmp(argv[i], "-e") == 0 || strcmp(argv[i], "-E") == 0) {
      showAPs = 0;
      continue;
    }
    if (strcmp(argv[i], "-n") == 0 || strcmp(argv[i], "-N") == 0) {
      onlyAddNew = 1;
      continue;
    }
    if (strcmp(argv[i], "-o") == 0 || strcmp(argv[i], "-O") == 0) {
      onlyAddOld = 1;
      continue;
    }
    temp = 0;
    if (strcmp(argv[i], "-l") == 0 || strcmp(argv[i], "-L") == 0) {
      temp = 1;
      i++;
      if (i >= argc) {
        printf ("-l requires that you specify an input csv file.\n");
        exit(1);
      }

    }
    dset = readCSVFile (argv[i], firstAp, firstEnddev, temp);
    firstAp = dset.s;
    firstEnddev = dset.e;
    numInputFiles++;
  }

  if (numInputFiles == 0) {
    printf ("Error: no input files specified.\n");
    exit(1);
  }
  if (textFile != NULL) {
    if (showAPs) printAPsToFileText (firstAp, textFile);
    if (showEnddevs) printEndDevicesToFileText (firstEnddev, textFile);
    fclose (textFile);
  }
  if (csvFile != NULL) {
    if (showAPs) {
      fprintf (csvFile, "BSSID, First time seen, Last time seen, channel, Speed, Privacy, Cipher, Authentication, ", CRLF);
      fprintf (csvFile, "Power, # beacons, # IV, LAN IP, ID-Length, ESSID, Key%s", CRLF);
      printAPsToFileCSV (firstAp, csvFile);
    }
    if (showEnddevs) {
      fprintf (csvFile, "Station MAC, First time seen, Last time seen, Power, # packets, BSSID, Probed ESSIDs%s", CRLF);
      printEndDevicesToFileCSV (firstEnddev, csvFile);
    }
    fclose (csvFile);
  }
  if (htmlFile != NULL) {
    if (showAPs) {
      fprintf (htmlFile, "<html>%s<head></head><body>%s<table border=\"1\">%s", CRLF, CRLF, CRLF);
      fprintf (htmlFile, "<tr><td>BSSID</td><td>Vendor</td><td>First time seen</td><td>Last time seen</td><td>channel</td><td>Speed</td><td>Privacy</td><td>Cipher</td><td>Authentication</td>", CRLF);
      fprintf (htmlFile, "<td>Power</td><td># beacons</td><td># IV</td><td>LAN IP</td><td>ID-Length</td><td>ESSID</td><td>Key</td></tr>%s", CRLF);
      printAPsToFileHTML (firstAp, htmlFile);
      fprintf (htmlFile, "</table>%s", CRLF);
    }
    if (showEnddevs) {
      fprintf (htmlFile, "<table border=\"1\">%s", CRLF);
      fprintf (htmlFile, "<tr><td>Station MAC</td><td>Vendor</td><td>First time seen</td><td>Last time seen</td><td>Power</td><td># packets</td><td>BSSID</td><td>ESSID</td><td>Probes</td></tr>%s", CRLF);
      printEndDevicesToFileHTML (firstEnddev, htmlFile);
      fprintf (htmlFile, "</table>%s</body>%s</html>", CRLF, CRLF, CRLF);
    }
    fclose (htmlFile);
  }

  free_ap(firstAp);
  free_enddev(firstEnddev);

  return 0;
}

